/**
 * (c) Meta Platforms, Inc. and affiliates. Confidential and proprietary.
 */


// 1. Navigator Object

const userAgent = navigator.userAgent;
const platform = navigator.platform;
const appVersion = navigator.appVersion;
const vendor = navigator.vendor;
const language = navigator.language;
const languages = navigator.languages ? navigator.languages.join(",") : null;
const onLine = navigator.onLine;
const cookieEnabled = navigator.cookieEnabled;
const hardwareConcurrency = navigator.hardwareConcurrency;
const maxTouchPoints = navigator.maxTouchPoints;
const deviceMemory = navigator.deviceMemory;


// 2. User Agent Parsing
function parseUserAgent(ua) {
    const isAndroid = /Android/i.test(ua);
    const isIOS = /iPhone|iPad|iPod/i.test(ua);
    const isMobile = /Mobile/i.test(ua);
    const browser = (() => {
    if (/CriOS/i.test(ua)) return "Chrome (iOS)";
    if (/Chrome/i.test(ua)) return "Chrome";
    if (/FxiOS/i.test(ua)) return "Firefox (iOS)";
    if (/Firefox/i.test(ua)) return "Firefox";
    if (/Safari/i.test(ua) && !/Chrome/i.test(ua)) return "Safari";
    if (/EdgiOS/i.test(ua)) return "Edge (iOS)";
    if (/EdgA/i.test(ua)) return "Edge (Android)";
    if (/Edge/i.test(ua)) return "Edge";
    return "Unknown";
    })();
    const os = isAndroid ? "Android" : isIOS ? "iOS" : "Other/Unknown";
    const firmwareMatch = ua.match(/Mobile\/(\w+)/);
    const firmware = firmwareMatch ? firmwareMatch[1] : null;
    const osMatch = ua.match(/OS ((\d+_?){2,3})\s/);
    const osVersion = osMatch ? osMatch[1] : null;
    return { isAndroid, isIOS, isMobile, browser, os, firmware, osVersion};
}
const uaInfo = parseUserAgent(navigator.userAgent);

const isAndroid = uaInfo.isAndroid;
const isIOS = uaInfo.isIOS;
const isMobile = uaInfo.isMobile;
const browser = uaInfo.browser;
const os = uaInfo.os;
const firmware = uaInfo.firmware;
const osVersion = uaInfo.osVersion;


// 3. Screen & Viewport
const width = window.screen.width;
const height = window.screen.height;
const availWidth = window.screen.availWidth;
const availHeight = window.screen.availHeight;
const innerWidth = window.innerWidth;
const innerHeight = window.innerHeight;
const devicePixelRatio = window.devicePixelRatio;
const orientationType = (screen.orientation && screen.orientation.type);
const orientationAngle = (screen.orientation && screen.orientation.angle);

// 4. Touch/Input Capabilities
const onTouchStart = 'ontouchstart' in window;
const isTouchDevice = 'ontouchstart' in window || navigator.maxTouchPoints > 0;
const pointerEvents = window.PointerEvent ? 'Supported' : 'Not Supported';


// 5. Mobile-Specific APIs
async function getBatteryInfo() {
  if (navigator.getBattery) {
    try {
      const battery = await navigator.getBattery();
      return {
        batteryCharging: battery.charging,
        batteryLevel: (battery.level * 100) + '%',
        batteryChargingTime: battery.chargingTime + 's',
        batteryDischargingTime: battery.dischargingTime + 's'
      };
    } catch (e) {
      // Fallback for rare error cases
      return {
        batteryCharging: null,
        batteryLevel: null,
        batteryChargingTime: null,
        batteryDischargingTime: null
      };
    }
  } else {
    // Battery API not supported
    return {
      batteryCharging: null,
      batteryLevel: null,
      batteryChargingTime: null,
      batteryDischargingTime: null
    };
  }
}
const {
    batteryCharging,
    batteryLevel,
    batteryChargingTime,
    batteryDischargingTime
} = await getBatteryInfo();

// 6. Network Information APIs
const connectionType = navigator.connection ? navigator.connection.effectiveType : null;
const downlink = navigator.connection ? navigator.connection.downlink + ' Mbps' : null;
const rtt = navigator.connection ? navigator.connection.rtt + ' ms' : null;
const saveData = navigator.connection ? navigator.connection.saveData : null;

// 7. Vibration API
const hasVibrator = navigator.vibrate ? 'Yes' : 'No';

// 8. Location / Timezone / Other APIs
const canvas = document.createElement('canvas');
const gl = canvas.getContext('webgl') || canvas.getContext('experimental-webgl');
var unmaskedVendor = null;
var unmaskedRenderer = null;
var maskedVendor = null;
var maskedRenderer = null;
if (gl) {
    maskedVendor = gl.getParameter(gl.VENDOR);
    maskedRenderer = gl.getParameter(gl.RENDERER);
    const debugInfo = gl.getExtension('WEBGL_debug_renderer_info');
    if (debugInfo) {
        unmaskedVendor = gl.getParameter(debugInfo.UNMASKED_VENDOR_WEBGL);
        unmaskedRenderer = gl.getParameter(debugInfo.UNMASKED_RENDERER_WEBGL);
    }
}
var textImage = getCanvasText();
textImage = textImage === 'N/A' ? null : textImage;
const geometryImage = getCanvasGeometryImage();

const timezone = Intl.DateTimeFormat().resolvedOptions().timeZone;
const plugins = getPlugins().join(', ');
const fonts = getFonts().join(', ');

// RETURN ALL DATA
return {
    'userAgent': toStringOrNull(userAgent),
    'platform': toStringOrNull(platform),
    'appVersion': toStringOrNull(appVersion),
    'vendor': toStringOrNull(vendor),
    'language': toStringOrNull(language),
    'languages': toStringOrNull(languages),
    'onLine': toStringOrNull(onLine),
    'cookieEnabled': toStringOrNull(cookieEnabled),
    'hardwareConcurrency': toStringOrNull(hardwareConcurrency),
    'maxTouchPoints': toStringOrNull(maxTouchPoints),
    'deviceMemory': toStringOrNull(deviceMemory),
    'isAndroid': toStringOrNull(isAndroid),
    'isIOS': toStringOrNull(isIOS),
    'isMobile': toStringOrNull(isMobile),
    'browser': toStringOrNull(browser),
    'os': toStringOrNull(os),
    'firmware': toStringOrNull(firmware),
    'osVersion': toStringOrNull(osVersion),
    'width': toStringOrNull(width),
    'height': toStringOrNull(height),
    'availWidth': toStringOrNull(availWidth),
    'availHeight': toStringOrNull(availHeight),
    'innerWidth': toStringOrNull(innerWidth),
    'innerHeight': toStringOrNull(innerHeight),
    'devicePixelRatio': toStringOrNull(devicePixelRatio),
    'orientationType': toStringOrNull(orientationType),
    'orientationAngle': toStringOrNull(orientationAngle),
    'onTouchStart': toStringOrNull(onTouchStart),
    'isTouchDevice': toStringOrNull(isTouchDevice),
    'pointerEvents': toStringOrNull(pointerEvents),
    'batteryCharging': toStringOrNull(batteryCharging),
    'batteryLevel': toStringOrNull(batteryLevel),
    'batteryChargingTime': toStringOrNull(batteryChargingTime),
    'batteryDischargingTime': toStringOrNull(batteryDischargingTime),
    'connectionType': toStringOrNull(connectionType),
    'downlink': toStringOrNull(downlink),
    'rtt': toStringOrNull(rtt),
    'saveData': toStringOrNull(saveData),
    'hasVibrator': toStringOrNull(hasVibrator),
    'timezone': toStringOrNull(timezone),
    'unmaskedVendor': toStringOrNull(unmaskedVendor),
    'unmaskedRenderer': toStringOrNull(unmaskedRenderer),
    'maskedVendor': toStringOrNull(maskedVendor),
    'maskedRenderer': toStringOrNull(maskedRenderer),
    'plugins': toStringOrNull(plugins),
    'fonts': toStringOrNull(fonts),
    'textImage': toStringOrNull(textImage),
    'geometryImage': toStringOrNull(geometryImage)
};

// Helper functions
function getPlugins() {
    if (navigator.plugins) {
    return Array.from(navigator.plugins).map(plugin => plugin.name);
    }
    return [];
}

function toStringOrNull(val) {
    if (val === null || val === undefined) return null;
    try {
        // Optionally, skip objects and arrays if you don't want '[object Object]' or similar
        if (typeof val === 'object') return null;
        return String(val);
    } catch {
        return null;
    }
}

function isFontAvailable(font) {
    const testString = "mmmmmmmmmmlli";
    const testSize = "72px";
    const canvas = document.createElement("canvas");
    const context = canvas.getContext("2d");
    // Fallback font
    context.font = `${testSize} monospace`;
    const baselineSize = context.measureText(testString).width;
    // Test font
    context.font = `${testSize} '${font}', monospace`;
    const newSize = context.measureText(testString).width;
    return newSize !== baselineSize;
}

function getFonts() {
    const fontsToTest = ["Arial", "Times New Roman", "Comic Sans MS", "Papyrus", "Courier New", "Impact", "Verdana", "Georgia", "Trebuchet MS", "Lucida Sans Unicode", "Palatino Linotype", "Garamond", "Tahoma", "Geneva", "Arial Black", "Lucida Console", "Courier", "Monaco", "Helvetica", "Helvetica Neue", "Optima", "Arial Narrow", "Calibri", "Segoe UI", "Segoe UI Emoji", "Segoe UI Symbol", "Candara", "Franklin Gothic Medium", "Century Gothic", "Roboto", "Open Sans", "Lato"];
    const availableFonts = fontsToTest.filter(isFontAvailable);
    return availableFonts;
}

function getCanvasText() {
    const canvas = document.createElement('canvas')
    canvas.width = 240
    canvas.height = 60
    const context = canvas.getContext("2d");
    context.textBaseline = 'alphabetic'
    context.fillStyle = '#f60'
    context.fillRect(100, 1, 62, 20)

    context.fillStyle = '#069'
    context.font = '11pt "Times New Roman"'
    const printedText = `Cwm fjordbank gly ${String.fromCharCode(55357, 56835) /* 😃 */}`
    context.fillText(printedText, 2, 15)
    context.fillStyle = 'rgba(102, 204, 0, 0.2)'
    context.font = '18pt Arial'
    context.fillText(printedText, 4, 45)

    const textImage1 = canvas.toDataURL();
    const textImage2 = canvas.toDataURL();

    if (textImage1 !== textImage2) {
    console.log('Canvas fingerprinting not stable!');
    return 'N/A';
    }
    return textImage1;
}

function getCanvasGeometryImage() {
    const canvas = document.createElement('canvas')
    const context = canvas.getContext("2d");
    // Resizing the canvas cleans it
    canvas.width = 122
    canvas.height = 110

    // Canvas blending
    context.globalCompositeOperation = 'multiply'
    for (const [color, x, y] of [
    ['#f2f', 40, 40],
    ['#2ff', 80, 40],
    ['#ff2', 60, 80],
    ]) {
    context.fillStyle = color
    context.beginPath()
    context.arc(x, y, 40, 0, Math.PI * 2, true)
    context.closePath()
    context.fill()
    }

    // Canvas winding
    context.fillStyle = '#f9c'
    context.arc(60, 60, 60, 0, Math.PI * 2, true)
    context.arc(60, 60, 20, 0, Math.PI * 2, true)
    context.fill('evenodd')

    return canvas.toDataURL();
}
